/** @file   bullet.h
 * @brief   Declaration of Bullet, RifleBullet.
 * @version $Revision: 1.3 $
 * @date    $Date: 2006/04/25 18:01:52 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_BULLET_H
#define H_WWW_BULLET_H

#include "eng2d.h"
#include "weapon.h"
#include "ammunition.h"
#include "weapon.h"

namespace WeWantWar {


// Just let the compiler know about this class.
class GameObject;


/** @class  Bullet
 * @brief   Represents the bullet the objects can shoot.
 * @author  Tomi Lamminsaari
 *
 */
class Bullet : public Ammunition
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================
  
  /** Available bullet types.
   */
  enum TType {
    ERifle,             // 0
    EShotgun,
    EGrenade,
    EInvisible,
    EAlienFireball,
    EFlameThrower,      // 5
    EMinigun,
    EUzi,
    ETankBullet,
    ETankFragments,
    ECarFragments,      // 10
    EInfantryMines,
    ECrowbar,
    ESentryFireball,
    ESniperRifle,
    ERocketLauncher,    // 15
    EAlienFireball2,
    EUnnamed
  };



  /** Returns the type code of the bullet that matches the given weapon.
   * @param     aWeapon           The weapon
   * @return    The typecode that matches the weapon. For instance when
   *            aWeapon is type Weapon::W_RIFLE we return BUllet::ERifle
   */
  TType findBulletType( const Weapon& aWeapon );
  
  /** Converts the given bullet type to Weapon-object.
   * @param     aType             Type of the Bullet.
   * @return    An instance of weapon that matches the given bullet. If there
   *            is no match, the @c id - field of this object has value
   *            Weapon::W_NOWEAPON
   */
  static Weapon bullet2Weapon( Bullet::TType aType );
  
  
  
  ///
  /// Public members
  ///
  
  /** The type of this bullet. */
  TType   iType;

  /** Origin of this bullet. The starting position. */
  eng2d::Vec2D iOrigin;
  
  /** The maximum range this bullet can fly. */
  float iRange;
  
  /** Amount of damage this bullet causes. */
  float iDamage;
  
  /** Size of the bullet in pixels */
  float iRadius;

  /** Drawing rotation. Number of degrees the Bullet must be rotated
   * when it's being drawn.
   */
  int iRotation;


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor. This constructor sets the default values to the member
   * variables based on the given type.
   * @param     aShooter          Pointer to GameObject who shot this.
   * @param     aBulletType       Type id of this new bullet.
   * @param     aPos              Starting position of this bullet.
   */
	Bullet( GameObject* aShooter, TType aBulletType, const eng2d::Vec2D& aPos );
	

	/** Destructor
   */
	virtual ~Bullet();

private:

	/** Copy constructor.
   * @param     aObject           Reference to another Bullet
   */
  Bullet( const Bullet& aObject );

	/** Assignment operator
   * @param     rO                Reference to another Bullet
   * @return    Reference to us.
   */
  Bullet& operator = ( const Bullet& rO );

public:


  ///
  /// Methods
  /// =======
  
  /** Updates this Bullet. This method manages bullets' flying ranges. If
   * the @c KTrails - flag is on, we also spawn the bullettrail animations.
   * @return    <code>true</code> if this Bullet should be destroyed.
   */
  virtual bool updateAmmunition();

  /** Draws this bullet to given bitmap.
   * @param     aTarget           Pointer to target bitmap
   */
  virtual void redraw( BITMAP* aTarget ) const;
  

  ///
  /// Getter methods
  /// ==============
  
  


protected:

  
  
  

  ///
  /// Members
  /// =======



private:

  ///
  /// Private members
  /// ===============

};


};  // end of namespace


#endif

/**
 * Version history
 * ===============
 * $Log: bullet.h,v $
 * Revision 1.3  2006/04/25 18:01:52  lamminsa
 * New bullet enum.
 *
 * Revision 1.2  2006/03/16 21:01:37  lamminsa
 * Only some comments changed.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.1  2005-11-13 14:36:22+02  lamminsa
 * RocketLauncher bullets added. The bullet class hierarchy was changed.
 *
 */
 
